#!/usr/bin/env python3
"""
🖥️ Windows Listener v2.7 - Render Proxy
========================================
Now uses Render proxy to bypass Oracle firewall
"""

import requests
import subprocess
import time
import json
import os
from datetime import datetime

# === CONFIG ===
# NEW: Using Render proxy instead of direct Oracle VM
API_URL = "https://wintask-proxy.onrender.com"
POLL_INTERVAL = 10
CLAUDE_CMD = "claude"

def log(msg):
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    print(f"[{timestamp}] {msg}")

def execute_claude(prompt, timeout=120):
    """Execute prompt via Claude Code CLI"""
    try:
        cmd = [CLAUDE_CMD, "-p", "--dangerously-skip-permissions", prompt]
        result = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=timeout,
            shell=True
        )
        output = result.stdout + result.stderr
        return {
            "success": result.returncode == 0,
            "output": output[:4000],
            "exit_code": result.returncode
        }
    except subprocess.TimeoutExpired:
        return {"success": False, "output": f"Timeout after {timeout}s", "exit_code": -1}
    except Exception as e:
        return {"success": False, "output": str(e), "exit_code": -1}

def execute_powershell(cmd):
    """Execute PowerShell command"""
    try:
        result = subprocess.run(
            ["powershell", "-Command", cmd],
            capture_output=True,
            text=True,
            timeout=60
        )
        return {
            "success": result.returncode == 0,
            "output": (result.stdout + result.stderr)[:4000],
            "exit_code": result.returncode
        }
    except Exception as e:
        return {"success": False, "output": str(e), "exit_code": -1}

def execute_python(code):
    """Execute Python code"""
    try:
        result = subprocess.run(
            ["python", "-c", code],
            capture_output=True,
            text=True,
            timeout=60
        )
        return {
            "success": result.returncode == 0,
            "output": (result.stdout + result.stderr)[:4000],
            "exit_code": result.returncode
        }
    except Exception as e:
        return {"success": False, "output": str(e), "exit_code": -1}

def process_task(task):
    """Process incoming task"""
    task_id = task.get("id")
    message = task.get("message", "")
    
    log(f"📥 Task {task_id}: {message[:50]}...")
    
    msg_upper = message.upper().strip()
    
    if msg_upper == "PING":
        return {"success": True, "output": "PONG! Listener v2.7 (Render) active", "exit_code": 0}
    
    if msg_upper == "STATUS":
        import platform
        info = f"OS: {platform.system()} {platform.release()}\nPython: {platform.python_version()}\nHost: {platform.node()}\nProxy: Render"
        return {"success": True, "output": info, "exit_code": 0}
    
    if msg_upper in ["STOP", "EXIT", "QUIT"]:
        log("🛑 Stop command received")
        return {"success": True, "output": "Listener stopping...", "exit_code": 0, "stop": True}
    
    if msg_upper == "RESTART":
        log("🔄 Restart command")
        os.system("start python listener_v2.py")
        return {"success": True, "output": "Restarting...", "exit_code": 0, "stop": True}
    
    if message.upper().startswith("PS:"):
        cmd = message[3:].strip()
        log(f"⚡ PowerShell: {cmd[:50]}")
        return execute_powershell(cmd)
    
    if message.upper().startswith("PY:"):
        code = message[3:].strip()
        log(f"🐍 Python: {code[:50]}")
        return execute_python(code)
    
    log(f"🤖 Claude Code: {message[:50]}")
    return execute_claude(message)

def main():
    log("=" * 50)
    log("🖥️ LISTENER v2.7 (Render Proxy)")
    log(f"API: {API_URL}")
    log("=" * 50)
    
    consecutive_errors = 0
    
    while True:
        try:
            response = requests.get(f"{API_URL}/task", timeout=30)
            
            if response.status_code == 200:
                task = response.json()
                if task.get("id"):
                    result = process_task(task)
                    
                    requests.post(
                        f"{API_URL}/result/{task['id']}",
                        json=result,
                        timeout=30
                    )
                    
                    log(f"✅ Task {task['id']} completed")
                    
                    if result.get("stop"):
                        log("👋 Goodbye!")
                        break
            
            consecutive_errors = 0
            
        except requests.exceptions.RequestException as e:
            consecutive_errors += 1
            log(f"⚠️ Connection error ({consecutive_errors}): {str(e)[:100]}")
            
            if consecutive_errors >= 10:
                log("❌ Too many errors, waiting 60s...")
                time.sleep(60)
                consecutive_errors = 0
        
        except Exception as e:
            log(f"❌ Error: {e}")
        
        time.sleep(POLL_INTERVAL)

if __name__ == "__main__":
    main()
